/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */
/*
 * Debug module.
 * If debug flags are not set, no code should be generated by DEBUG macros.
 */

#ifndef _GTOF_DEBUG_H_
#define _GTOF_DEBUG_H_

#include <fsl_common.h>
#include <stdio.h>

#if defined(__cplusplus)
extern "C" {
#endif // __cplusplus

///////////////////////////////////////////////////////////////////////////////////////////////////
// Flags to control debug options
///////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef GTOF_DEBUG_RAM_ON
#define GTOF_DEBUG_RAM_ON (0)
#endif

#ifndef GTOF_DEBUG_PIN_ON
#define GTOF_DEBUG_PIN_ON (0)
#endif

#ifndef GTOF_DEBUG_SYSVIEW_ON
#define GTOF_DEBUG_SYSVIEW_ON (0)
#endif

#ifndef GTOF_DEBUG_PRT_ON
#define GTOF_DEBUG_PRT_ON (0)
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////
// TERMINAL (semihosting) debugging section
///////////////////////////////////////////////////////////////////////////////////////////////////

#if (GTOF_DEBUG_PRT_ON == 1)
#define GTOF_DEBUG_PRT(...) (void)printf(__VA_ARGS__)
#define GTOF_ERROR(...) (void)printf(__VA_ARGS__)
#else
#define GTOF_DEBUG_PRT(...)
#define GTOF_ERROR(...)
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////
// SYSVIEW debugging section
///////////////////////////////////////////////////////////////////////////////////////////////////

/*
 * List of trace point, used in cunjunction with sysview config file, based on OS name passed to SEGGER_SYSVIEW_SendSysDesc()
 * C:\Program Files (x86)\SEGGER\SystemView_V240c\Description\ SYSVIEW_FreeRTOS.txt
 * C:\Program Files (x86)\SEGGER\SystemView_V240c\Description\ SYSVIEW_baremetal.txt
 */    
#define GTOF_DEBUG_EVT_PROCESS_EVT              0x1000
#define GTOF_DEBUG_EVT_STATE_TRANSITION         0x1001
#define GTOF_DEBUG_EVT_UNEXP_TRANSITION         0x10FF

#define GTOF_DEBUG_EVT_START_TX                 0x2000
#define GTOF_DEBUG_EVT_TX_BUFFER                0x2001
#define GTOF_DEBUG_EVT_START_RX                 0x2010
#define GTOF_DEBUG_EVT_RX_BUFFER                0x2011

#define GTOF_DEBUG_EVT_RX_ERROR                 0x3010
#define GTOF_DEBUG_EVT_RXTIMEOUT                0x3100
#define GTOF_DEBUG_EVT_ABORTTIMEOUT             0x3101
#define GTOF_DEBUG_EVT_MEASURETIMEOUT           0x3102

#define GTOF_DEBUG_EVT_START_MEASURE            0x4000
#define GTOF_DEBUG_EVT_DEVICE_STOP              0x4FFF

/* Enf of trace points */

#if (GTOF_DEBUG_SYSVIEW_ON == 1)
#include "SEGGER_SYSVIEW.h"
    
void GTOF_DbgSysviewInit(void);
void GTOF_DbgSysviewBuffer(uint32_t eventId, const uint8_t *pBuffer, uint32_t buflen);

#define GTOF_DEBUG_SYSVIEW_INIT() GTOF_DbgSysviewInit()
#define GTOF_DEBUG_EVENT(EVENTID) SEGGER_SYSVIEW_RecordVoid(EVENTID)
#define GTOF_DEBUG_VALUE(EVENTID, V1) SEGGER_SYSVIEW_RecordU32(EVENTID,V1)
#define GTOF_DEBUG_VALUE2(EVENTID, V1, V2) SEGGER_SYSVIEW_RecordU32x2(EVENTID,V1,V2)
#define GTOF_DEBUG_BUFFER(EVENTID, BUFFER, BUFLEN) GTOF_DbgSysviewBuffer(EVENTID, BUFFER, BUFLEN)

#elif (GTOF_DEBUG_RAM_ON == 1)

void GTOF_DbgRamInit(void);
void GTOF_DbgRamEvent(uint16_t eventId);
void GTOF_DbgRamValue(uint16_t eventId, uint32_t value);
void GTOF_DbgRamValue2(uint16_t eventId, uint32_t value1, uint32_t value2);
void GTOF_DbgRambuffer(uint16_t eventId, uint8_t *pBuffer, uint8_t bufLen);

#define GTOF_DEBUG_SYSVIEW_INIT() GTOF_DbgRamInit()
#define GTOF_DEBUG_EVENT(EVENTID) GTOF_DbgRamEvent((EVENTID))
#define GTOF_DEBUG_VALUE(EVENTID, V1) GTOF_DbgRamValue((EVENTID),(uint32_t)(V1))
#define GTOF_DEBUG_VALUE2(EVENTID, V1, V2) GTOF_DbgRamValue2((EVENTID),(uint32_t)(V1),(uint32_t)(V2))
#define GTOF_DEBUG_BUFFER(EVENTID, BUFFER, BUFLEN) GTOF_DbgRambuffer((EVENTID), (BUFFER), (BUFLEN))

#else

#define GTOF_DEBUG_SYSVIEW_INIT()
#define GTOF_DEBUG_EVENT(EVENTID)
#define GTOF_DEBUG_VALUE(EVENTID, V1)
#define GTOF_DEBUG_VALUE2(EVENTID, V1, V2)
#define GTOF_DEBUG_BUFFER(EVENTID, BUFFER, BUFLEN)

#endif /* GTOF_DEBUG_SYSVIEW_ON */


///////////////////////////////////////////////////////////////////////////////////////////////////
// GPIO debugging section
///////////////////////////////////////////////////////////////////////////////////////////////////

#if (GTOF_DEBUG_PIN_ON == 1)
#include "GPIO_Adapter.h"

void GTOF_DbgGpioInit(void);
#define GTOF_DEBUG_IO_INIT() GTOF_DbgGpioInit()

extern const gpioOutputPinConfig_t gtof_dbg_io_pins[];

static inline void GTOF_DbgSetIo(int pin_id, int val)
{
    /* Wrapper to control gtof_dbg_io_pins */
    if (val != 0)
    {
        GpioSetPinOutput(&gtof_dbg_io_pins[pin_id]);
    }
    else
    {
        GpioClearPinOutput(&gtof_dbg_io_pins[pin_id]);
    }
}
#define GTOF_DEBUG_IO_SET(pinId) GTOF_DbgSetIo(pinId, 1)
#define GTOF_DEBUG_IO_CLEAR(pinId) GTOF_DbgSetIo(pinId, 0)
#define GTOF_DEBUG_IO_SETV(pinId, value) GTOF_DbgSetIo(pinId, value)

static inline void TOF_DbgToggleIo(int pin_id)
{
    /* Wrapper to control gtof_dbg_io_pins */
    GpioTogglePinOutput(&gtof_dbg_io_pins[pin_id]);
}
#define GTOF_DEBUG_IO_TOGGLE(pinId) TOF_DbgToggleIo(pinId)

#else /* GTOF_DEBUG_PIN_ON */

#define GTOF_DEBUG_IO_INIT()

#define GTOF_DEBUG_IO_SET(pinId)
#define GTOF_DEBUG_IO_CLEAR(pinId)
#define GTOF_DEBUG_IO_TOGGLE(pinId)
#define GTOF_DEBUG_IO_SETV(pinId, value)

#endif /* GTOF_DEBUG_PIN_ON */

///////////////////////////////////////////////////////////////////////////////////////////////////
// General debugging section
///////////////////////////////////////////////////////////////////////////////////////////////////

#define GTOF_DEBUG_INIT() \
    do { \
        GTOF_DEBUG_SYSVIEW_INIT(); \
        GTOF_DEBUG_IO_INIT(); \
    } while (FALSE);


#if defined(__cplusplus)
}
#endif // __cplusplus


#endif /* _DEBUG_H_ */
///////////////////////////////////////////////////////////////////////////////////////////////////
// EOF
///////////////////////////////////////////////////////////////////////////////////////////////////
